/* FILE: mhalfsp.c                              (D. Tottingham  10/13/90)

These C functions implement the halfspace location algorithm developed by
John Lahr et. al.  All functions have been written and compiled medium model.
The following functions are included:

hs_initialize ()                initialize halfspace module
hs_locate_event ()              locate event using halfspace algorithm
hs_set_CriticalPhi ()           set critical phi
hs_set_HalfspaceVelocity ()     set halfspace velocity

EXTERNAL FUNCTIONS CALLED:

er_abort ()                     display an error message then quit
ll_to_latlon ()                 convert x, y to lat, lon
pk_get_first_arrival ()         get first p-arrival
pk_get_first_magnitude ()       get first magnitude
pk_get_last_arrival ()          get last p-arrival
pk_get_last_magnitude ()        get last magnitude
pk_get_next_arrival ()          get next p-arrival
pk_get_next_magnitude ()        get next magnitude
pk_get_narrivals ()             get number of p-arrivals
pk_get_nmagnitudes ()           get number of magnitudes
st_get_station ()               get station from station queue

HISTORY:

V1.95   (09/29/90)  Added rms residual calculation to hs_locate_event().


*/


/*************************************************************************
                            INCLUDE FILES

*************************************************************************/
#include <malloc.h>
#include <math.h>
#include <stdio.h>

#include "mconst.h"
#include "merror.h"
#include "mhalfsp.h"
#include "mlatlon.h"
#include "mpick.h"
#include "mstation.h"
#include "xdetect.h"
#include "suds.h"


/*************************************************************************
                              GLOBAL DATA

**************************************************************************/
PRIVATE unsigned int critical_phi;
PRIVATE double hs_velocity, hs_vsq, hsv_inverted;


/*=======================================================================*
 *                               deter3x3                                *
 *=======================================================================*/
/* Find 3 x 3 determinant of 4 x 4 matrix.  Parameter matrix is a 4 x 4
   matrix.  Parameters col1, col2, col3 are the columns of the 4 x 4
   matrix to be used to compute the determinant.                         */

PRIVATE
double deter3x3 (matrix, col0, col1, col2)
double * * matrix;
unsigned int col0, col1, col2;
{
   double deter;

   deter = ( matrix[col0][0] * ( matrix[col1][1] * matrix[col2][2] -
           matrix[col2][1] * matrix[col1][2] )) -
           ( matrix[col1][0] * ( matrix[col0][1] * matrix[col2][2] -
           matrix[col2][1] * matrix[col0][2] )) +
           ( matrix[col2][0] * ( matrix[col0][1] * matrix[col1][2] -
           matrix[col1][1] * matrix[col0][2] ));
   return (deter);
}

/*=======================================================================*
 *                            hs_initialize                              *
 *=======================================================================*/
/* Initialize the halfspace module.                                      */

PUBLIC
void hs_initialize ()
{
   critical_phi = CRITICAL_PHI;
   hs_velocity = HALFSPACE_VELOCITY;
   hs_vsq = hs_velocity * hs_velocity;
   hsv_inverted = 1.0 / hs_velocity;
}

/*=======================================================================*
 *                           hs_locate_event                             *
 *=======================================================================*/
/* Locate event using halfspace algorithm.                               */

PUBLIC
FLAG hs_locate_event (origin)
SUDS_ORIGIN far * origin;
{
   Q_ARRIVAL far *pick_ptr, far *first_pick, far *last_pick;
   Q_MAGNITUDE far *mag_ptr;
   Q_STATION far *station_ptr, far *last_station;
   double *sym_tensor[4], *normal_eqn[4], d, x, y, rel_origin;
   double deltat, deltax, deltay, depthsq, coda_magnitude, residual;
   unsigned int n_arrivals, n_magnitudes, i, j, k;
   LL_LATLON latlon;
   FLAG located_flag, first_time;

   /* Do we have enough p-arrivals? */
   if ((n_arrivals = pk_get_narrivals()) < critical_phi) return (FALSE);

   /* Allocate some storage */
   for (i = 0; i < 4; i++) {
      sym_tensor[i] = (double *) malloc (4 * sizeof(double));
      normal_eqn[i] = (double *) malloc (n_arrivals * sizeof(double));
      if (sym_tensor[i] == NULL || normal_eqn[i] == NULL) er_abort (HS_NO_STORAGE);
   }

   /* Setup normal equations */
   first_pick = pk_get_first_arrival ();
   last_pick = pk_get_last_arrival ();
   last_station = st_get_station (last_pick->channel);
   for (i = 0, pick_ptr = first_pick; i < (n_arrivals - 1); i++,
               pick_ptr = pk_get_next_arrival()) {
      station_ptr = st_get_station (pick_ptr->channel);
      normal_eqn[0][i] = last_station->x - station_ptr->x;
      normal_eqn[1][i] = last_station->y - station_ptr->y;
      normal_eqn[2][i] = -(hs_vsq) * (last_pick->rel_arrival - pick_ptr->rel_arrival);
      normal_eqn[3][i] = (last_station->xsq + last_station->ysq -
                          (last_pick->rel_arrival * last_pick->rel_arrival * hs_vsq) -
                          station_ptr->xsq - station_ptr->ysq +
                          (pick_ptr->rel_arrival * pick_ptr->rel_arrival * hs_vsq)) / 2.0;
   }

/* Display normal equations
   if (Debug_enabled) {
      printf ("Normal equations:\n");
      for (i = 0; i < (n_arrivals - 1); i++) {
         printf ("%3d  ", i);
         for (j = 0; j < 4; j++)
            printf ("%10.6lf  ", normal_eqn[j][i]);
         printf ("\n");
      }
   }
*/

   /* Compute sum of squares matrix from coefficients */
   for (i = 0; i < 4; i++)
      for (j = i; j < 4; j++) {
         sym_tensor[i][j] = 0.0;
         for (k = 0; k < (n_arrivals - 1); k++)
            sym_tensor[i][j] += normal_eqn[i][k] * normal_eqn[j][k];
         sym_tensor[j][i] = sym_tensor[i][j];
      }

/* Display symmetric tensor
   if (Debug_enabled) {
      printf ("Symmetric tensor:\n");
      for (i = 0; i < 4; i++) {
         printf ("%3d  ", i);
         for (j = 0; j < 4; j++)
            printf ("%10.6f  ", sym_tensor[i][j]);
         printf ("\n");
      }
   }
*/

   /* Solve simultaneous equations using determinants */
   d = deter3x3 (sym_tensor, 0, 1, 2);

/*
   if (Debug_enabled) printf ("d = %10.5lf\n", d);
*/

   if (d > DETERMINANT_LIMIT) {
      x = deter3x3 (sym_tensor, 3, 1, 2) / d;
      y = deter3x3 (sym_tensor, 0, 3, 2) / d;
      rel_origin = deter3x3 (sym_tensor, 0, 1, 3) / d;

      /* Convert x, y to lat, lon */
      latlon = ll_to_latlon (y, -x, last_station->mean_latlon);
      origin->or_lat = latlon.lat;
      origin->or_long = -latlon.lon;

      /* Calculate origin time */
      origin->orgtime = rel_origin + first_pick->info.time;

      /* Calculate depth */
      deltat = last_pick->rel_arrival - rel_origin;
      deltax = last_station->x - x;
      deltay = last_station->y - y;
      depthsq = hs_vsq*deltat*deltat - deltax*deltax - deltay*deltay;
      if (depthsq > 0.0)
         origin->depth = sqrt (depthsq);
      else origin->depth = 0.0;

/*
      if (Debug_enabled)
         printf ("x, y, z, t: %lf, %lf, %lf, %lf\n", x, y, origin->depth, rel_origin);
*/

      /* Some fields in origin structure need filling in */
      origin->num_stats = origin->rep_p = origin->used_p = n_arrivals;
      origin->rep_s = origin->used_s = 0;

      /* Calculate magnitude */
      first_time = TRUE;
      if ((n_magnitudes = pk_get_nmagnitudes()) != 0) {
         for (mag_ptr = pk_get_first_magnitude(); mag_ptr != NULL; mag_ptr = pk_get_next_magnitude()) {
            coda_magnitude = -0.87 + (2.0 * log10(mag_ptr->info.time));
            if (first_time) {
               origin->magnitude = (float) coda_magnitude;
               first_time = FALSE;
            } else if (coda_magnitude > origin->magnitude)
               origin->magnitude = (float) coda_magnitude;
         }
         origin->rep_m = origin->used_m = n_magnitudes;
         origin->mag_type = 'c';
      }

      /* Calculate rms of residuals */
      origin->res_rms = 0.0;

      for (i = 0, pick_ptr = pk_get_first_arrival (); pick_ptr != NULL; i++,
                  pick_ptr = pk_get_next_arrival()) {
         station_ptr = st_get_station (pick_ptr->channel);
         residual = (station_ptr->x - x) * (station_ptr->x - x);
         residual += (station_ptr->y - y) * (station_ptr->y - y);
         residual += origin->depth * origin->depth;
/** debug **/
if (Debug_enabled)
   printf ("i, res calc1 = %d, %lf\n", i, residual);

         residual = -(hsv_inverted * sqrt(residual));
/** debug **/
if (Debug_enabled)
   printf ("i, res calc2 = %d, %lf\n", i, residual);

         residual += pick_ptr->rel_arrival - rel_origin;
/** debug **/
if (Debug_enabled)
   printf ("i, res calc3 = %d, %lf\n", i, residual);

         origin->res_rms += ((float)(residual * residual));
/** debug **/
if (Debug_enabled)
   printf ("i, rms calc1 = %d, %f\n", i, origin->res_rms);

      }
      origin->res_rms = sqrt(((double)(origin->res_rms)) / n_arrivals);

/** debug **/
if (Debug_enabled)
   printf ("res_rms = %f\n", origin->res_rms);


      located_flag = TRUE;
   } else located_flag = FALSE;

   /* Free storage */
   for (i = 0; i < 4; i++) {
      free (sym_tensor[i]); free (normal_eqn[i]);
   }
   return (located_flag);
}

/*=======================================================================*
 *                         hs_set_CriticalPhi                            *
 *=======================================================================*/
/* Set critical phi.                                                     */

PUBLIC
void hs_set_CriticalPhi (cp)
unsigned int cp;
{
   critical_phi = cp;
}

/*=======================================================================*
 *                       hs_set_HalfspaceVelocity                        *
 *=======================================================================*/
/* Set halfspace velocity.                                               */

PUBLIC
void hs_set_HalfspaceVelocity (velocity)
double velocity;
{
   hs_velocity = velocity;
   hs_vsq = hs_velocity * hs_velocity;
   hsv_inverted = 1.0 / hs_velocity;
}
